# -*- coding: utf-8 -*-
import os
import re
import time
import json
import threading
import requests
import urllib3
import tkinter as tk
from tkinter import filedialog, messagebox, ttk, scrolledtext, font
from datetime import datetime

# =============================================================================
# ⭐ 基础配置
# =============================================================================

DEFAULT_API_URL = "http://api.example.com/v1/chat/completions" 
DEFAULT_API_KEY = "sk-xxxxx"
DEFAULT_MODEL_NAME = "xxxxx"
DEFAULT_MAX_TOKENS = 800

# =============================================================================
# 🌍 国际化资源字典 (I18N) - 12 Language Support
# =============================================================================

UI_STRINGS = {
    'zh': {
        'app_title': "佛典AI译丛 (BCAITS) 禅宗公案·完美对称系统 v14.0",
        'group_api': "API 与 模型配置",
        'lbl_url': "API 地址:",
        'lbl_key': "API Key:",
        'lbl_model': "模型名称:",
        'group_file': "输入源 (支持全球多语言文本)",
        'btn_select_file': "📂 选择文件",
        'lbl_threshold': "分段阈值 (Tokens):",
        'group_lang': "任务矩阵 (自动生成：split + 独立文件 + 对照版)", 
        'col_lang': "语言",
        'col_trans': "基础翻译",
        'col_exe': "深度疏通与解经",
        'group_log': "系统日志 (进度)",
        'group_preview': "实时预览 (内容)",
        'btn_start': "🚀 启动任务",
        'btn_stop': "🛑 停止",
        'msg_err_nofile': "错误：请先选择输入文件",
        'msg_err_notask': "错误：请至少勾选一项任务",
        'msg_stop': "⚠️ 正在停止...",
        'msg_read_start': "读取文本中...",
        'msg_seg_done': "✅ 智能分段完成：共 {} 个片段。",
        'msg_split_created': "已生成原始分段文件: {}",
        'msg_file_created': "创建文件: {}",
        'msg_processing': "处理片段 ({}/{}): {}",
        'msg_generating': "  -> [{}] 生成 {} ...",
        'msg_done_title': "完成",
        'msg_done_body': "处理完成！\n已生成 [split] 原文、独立文件及 [对照版] 文件。",
        'err_fatal': "❌ 错误: {}",
        # 语言列表
        'lang_zh': "中文", 'lang_en': "英文", 'lang_ja': "日文",
        'lang_ko': "韩文", 'lang_bo': "藏文", 'lang_pali': "巴利文",
        'lang_es': "西班牙语", 'lang_ru': "俄语", 'lang_fr': "法语", 'lang_de': "德语",
        'lang_hi': "印地语", 'lang_no': "挪威语",
    },
    'en': {
        'app_title': "BCAITS Zen Symmetric System v14.0",
        'group_api': "API & Model Settings",
        'lbl_url': "API Endpoint:",
        'lbl_key': "API Key:",
        'lbl_model': "Model Name:",
        'group_file': "Input Source",
        'btn_select_file': "📂 Browse...",
        'lbl_threshold': "Threshold:",
        'group_lang': "Task Matrix (Auto-generates Contrast File)",
        'col_lang': "Language",
        'col_trans': "Translation",
        'col_exe': "Guide & Exegesis",
        'group_log': "Log",
        'group_preview': "Preview",
        'btn_start': "🚀 EXECUTE",
        'btn_stop': "🛑 STOP",
        'msg_err_nofile': "Error: Select file.",
        'msg_err_notask': "Error: Select at least one task.",
        'msg_stop': "⚠️ Stopping...",
        'msg_read_start': "Reading text...",
        'msg_seg_done': "✅ Segments: {}.",
        'msg_split_created': "Created split file: {}",
        'msg_file_created': "Created: {}",
        'msg_processing': "Processing ({}/{}): {}",
        'msg_generating': "  -> [{}] Generating {} ...",
        'msg_done_title': "Finished",
        'msg_done_body': "Done! Split file, Separate files and Contrast files generated.",
        'err_fatal': "❌ Error: {}",
        'lang_zh': "Chinese", 'lang_en': "English", 'lang_ja': "Japanese",
        'lang_ko': "Korean", 'lang_bo': "Tibetan", 'lang_pali': "Pali",
        'lang_es': "Spanish", 'lang_ru': "Russian", 'lang_fr': "French", 'lang_de': "German",
        'lang_hi': "Hindi", 'lang_no': "Norwegian",
    },
    'ja': {'app_title': "BCAITS v14.0", 'group_api':"API設定", 'lbl_url':"URL:", 'lbl_key':"Key:", 'lbl_model':"Model:", 'group_file':"入力", 'btn_select_file':"選択", 'lbl_threshold':"閾値:", 'group_lang':"タスク (対照版を自動生成)", 'col_lang':"言語", 'col_trans':"翻訳", 'col_exe':"解説・解釈", 'group_log':"ログ", 'group_preview':"プレビュー", 'btn_start':"🚀 開始", 'btn_stop':"🛑 停止", 'msg_err_nofile':"ファイル未選択", 'msg_err_notask':"タスク未選択", 'msg_stop':"停止中...", 'msg_read_start':"読込中...", 'msg_seg_done':"完了: {}", 'msg_split_created':"分割ファイル: {}", 'msg_file_created':"作成: {}", 'msg_processing':"処理中 ({}/{}): {}", 'msg_generating':"-> [{}] 生成 {}", 'msg_done_title':"完了", 'msg_done_body':"完了", 'err_fatal':"エラー: {}", 'lang_zh':"中国語", 'lang_en':"英語", 'lang_ja':"日本語", 'lang_ko':"韓国語", 'lang_bo':"チベット語", 'lang_pali':"パーリ語", 'lang_es':"スペイン語", 'lang_ru':"ロシア語", 'lang_fr':"フランス語", 'lang_de':"ドイツ語", 'lang_hi':"ヒンディー語", 'lang_no':"ノルウェー語"},
}

# =============================================================================
# 🧠 AI 核心配置：多语种Prompt (12种)
# =============================================================================

class LangConfig:
    @staticmethod
    def get_trans_prompt(lang_code):
        base = "You are a professional translator of Buddhist texts."
        rules = "Translate the following Zen Koan accurately and smoothly. No commentary. Keep the original title."
        
        prompts = {
            'zh': f"{base} 请将公案翻译为通俗流畅的现代白话中文。{rules}",
            'en': f"{base} Translate into elegant English. {rules}",
            'ja': f"{base} 現代日本語に翻訳してください。{rules}",
            'ko': f"{base} 현대 한국어로 번역하십시오. {rules}",
            'bo': f"{base} Translate into Tibetan. {rules}",
            'pali': f"{base} Translate into Romanized Pali. No Devanagari. {rules}",
            'es': f"{base} Translate into clear and elegant Spanish. {rules}",
            'ru': f"{base} Translate into literary and precise Russian. {rules}",
            'fr': f"{base} Translate into refined French. {rules}",
            'de': f"{base} Translate into precise German (philosophical tone). {rules}",
            
            # New Languages
            'hi': f"{base} Translate into formal Hindi (Devanagari script). {rules}",
            'no': f"{base} Translate into clear Norwegian (Bokmål). {rules}",
        }
        return prompts.get(lang_code, prompts['en'])

    @staticmethod
    def get_exe_prompt(lang_code):
        CORE_INSTRUCTION = """
        [Role]: You are a gentle, patient Zen teacher, well-versed in Buddhist Doctrine (Yogacara/Madhyamaka) and Cognitive Science.
        [View]: 
        - **Doctrine & Science**: When encountering complex imagery of practice/realization (修证意象), you MUST combine **Buddhist Doctrine** (e.g., Three Natures of Yogacara, Two Truths) AND **Cognitive Science** (e.g., Embodied Cognition, Neural Correlates of Consciousness, Predictive Coding) to elucidate the experience.
        - **Spontaneous Mirrors**: Use Madhyamaka logic or Quantum metaphors only when they naturally fit.
        [Style]: Detailed textual clarification; Profound yet accessible analysis of mental states.
        [Structure Required]:
        1. Historical Context (Detailed)
        2. Textual Clarification (Detailed breakdown)
        3. **Cognitive & Doctrinal Analysis**: (Specifically analyzing the imagery of realization using the combined view mentioned above).
        4. The Gentle Point or Verse.
        *Vary your style slightly each time.*
        """
        
        prompts = {
            'zh': f"""{CORE_INSTRUCTION}
            输出格式：
            【公案】[标题]
            【背景情景】[时代、人物、来龙去脉]
            【文字疏通】[关键词解释、句意逐层拆解]
            【修证解析】[重点：结合教理（如唯识/中观）与认知科学（如具身认知/元认知）解析公案中的修证意象与心理机制]
            【直指/偈颂】[温和点醒或总结]""",
            
            'en': f"""{CORE_INSTRUCTION} Output: [Koan]; [Context]; [Clarification]; [Cognitive & Doctrinal Analysis]; [Point].""",
            'ja': f"""{CORE_INSTRUCTION} 出力: 【公案】; 【背景】; 【語句解説】; 【修証解析】(教理と認知科学); 【直指・偈】。""",
            'ko': f"""{CORE_INSTRUCTION} 출력: 【공안】; 【배경】; 【문구 소통】; 【수증 해석】(교리와 인지과학); 【직지/게송】。""",
            'bo': f"""{CORE_INSTRUCTION} Output: 【Gzhung】; 【rGyu-mtshan】; 【gSal-byed】; 【Nyams-len dPyad-pa】; 【Nying-po】.""",
            'pali': f"""{CORE_INSTRUCTION} Roman Only. Output: 【Nidāna】; 【Padavanṇanā】; 【Bhāvanā-Vicāra】; 【Sāra】.""",
            'es': f"""{CORE_INSTRUCTION} Output (Spanish): [Contexto]; [Clarificación]; [Análisis Cognitivo y Doctrinal]; [Punto].""",
            'ru': f"""{CORE_INSTRUCTION} Output (Russian): [Контекст]; [Разъяснение]; [Когнитивный и Доктринальный Анализ]; [Суть].""",
            'fr': f"""{CORE_INSTRUCTION} Output (French): [Contexte]; [Clarification]; [Analyse Cognitive et Doctrinale]; [Point].""",
            'de': f"""{CORE_INSTRUCTION} Output (German): [Kontext]; [Klärung]; [Kognitive und Doktrinäre Analyse]; [Punkt].""",
            
            # New Languages Exegesis Prompts
            'hi': f"""{CORE_INSTRUCTION}
            Output Format (Hindi):
            [Koan] [शीर्षक]
            [ऐतिहासिक संदर्भ] (Historical Context)
            [पाठ्य स्पष्टीकरण] (Textual Clarification)
            [संज्ञानात्मक और सिद्धांतिक विश्लेषण] (Cognitive & Doctrinal Analysis - Combine Buddhist Doctrine & Science)
            [सारांश/श्लोक] (The Gentle Point)""",
            
            'no': f"""{CORE_INSTRUCTION}
            Output Format (Norwegian):
            [Koan] [Tittel]
            [Historisk Kontekst]
            [Tekstforklaring]
            [Kognitiv og Doktrinær Analyse] (Kombiner buddhistisk lære og kognitiv vitenskap)
            [Det Milde Poenget/Vers]"""
        }
        return prompts.get(lang_code, prompts['en'])

    @staticmethod
    def get_file_suffix(lang_code, task_type):
        file_lang_code = 'cn' if lang_code == 'zh' else lang_code
        
        names = {
            'trans': {
                'zh': '白话翻译', 'en': 'Translation', 'ja': '現代語訳', 
                'ko': '현대어번역', 'bo': 'Yig-sgyur', 'pali': 'Parivattana',
                'es': 'Traducción', 'ru': 'Perevod', 'fr': 'Traduction', 'de': 'Übersetzung',
                'hi': 'Anuvad', 'no': 'Oversettelse'
            },
            'exe': {
                'zh': '深度解经', 'en': 'Exegesis', 'ja': '深層解釈', 
                'ko': '심층평석', 'bo': 'Grel-bshad', 'pali': 'Vannana',
                'es': 'Exégesis', 'ru': 'Tolkovanie', 'fr': 'Exégèse', 'de': 'Auslegung',
                'hi': 'Vyakhyana', 'no': 'Tolkning'
            },
            'combined': {
                'zh': '中文对照版', 'en': 'Contrast', 'ja': '対訳版', 
                'ko': '대조판', 'bo': 'Shan-sbyar', 'pali': 'Samagga',
                'es': 'Contraste', 'ru': 'Kontrast', 'fr': 'Contraste', 'de': 'Kontrast',
                'hi': 'Contrast', 'no': 'Kontrast'
            }
        }
        
        func_name = names.get(task_type, {}).get(lang_code, 'Output')
        return f"_{func_name}.{file_lang_code}.txt"

    @staticmethod
    def validate_pali(text):
        clean_text = re.sub(r'[（(][^)）]*[)）]', '', text.replace('\ufeff', ''))
        if re.search(r'[\u0900-\u097F]', clean_text): return False, "Detected Devanagari"
        return True, None

# =============================================================================
# 📜 智能分段
# =============================================================================

class ZenTextProcessor:
    def __init__(self):
        self.header_pattern = re.compile(r'^\s*([^\s]+(?:禅师|和尚|大士|尊者|如来|佛)|[^\s]+(?:卷第.+)|Case\s+\d+|Chapter\s+\d+)\s*$')
        self.sentence_end_pattern = re.compile(r'([。！？；.!?;།༎।]+)')

    def preprocess_text(self, full_text):
        if len(full_text) > 1000 and full_text.count('\n') < 10:
            full_text = self.sentence_end_pattern.sub(r'\1\n', full_text)
        return full_text

    def smart_segmentation(self, full_text, max_chars=3000):
        full_text = self.preprocess_text(full_text)
        lines = full_text.split('\n')
        segments = []
        current_title = "Excerpt / 选段"
        current_buffer = []
        current_count = 0
        is_standard_koan = (sum(1 for line in lines[:100] if self.header_pattern.match(line)) > 2)

        for line in lines:
            line = line.rstrip()
            if not line: continue
            should_split = False; new_title = None
            is_header = False
            
            if is_standard_koan:
                if self.header_pattern.match(line) and len(line) < 40: is_header = True
            else:
                if len(line) < 40 and not self.sentence_end_pattern.search(line[-1:]):
                    if current_count > 100: is_header = True
            
            if is_header: should_split = True; new_title = line.strip()
            if current_count + len(line) > max_chars: should_split = True; new_title = new_title if new_title else current_title + " (Part II)"

            if should_split and current_buffer:
                segments.append({"title": current_title, "content": "\n".join(current_buffer)})
                if new_title:
                    current_title = new_title
                    if is_header: current_buffer = []; current_count = 0
                    else: current_buffer = [line]; current_count = len(line)
                else: current_buffer = [line]; current_count = len(line)
            else:
                if is_header: current_title = line.strip()
                else: current_buffer.append(line); current_count += len(line)
        
        if current_buffer: segments.append({"title": current_title, "content": "\n".join(current_buffer)})
        if len(segments) > 1 and len(segments[-1]['content']) < (max_chars * 0.1):
            segments[-2]['content'] += "\n\n" + segments[-1]['content']; segments.pop()
        return segments

# =============================================================================
# 🤖 AI 引擎
# =============================================================================

class AiEngine:
    def __init__(self, api_url, api_key):
        self.api_url = api_url
        self.api_key = api_key
    def process(self, title, content, system_prompt, model_name, validator=None):
        user_prompt = f"Target Text: {title}\n\nContent:\n{content}"
        messages = [{"role": "system", "content": system_prompt}, {"role": "user", "content": user_prompt}]
        headers = {"Authorization": f"Bearer {self.api_key}", "Content-Type": "application/json"}
        for _ in range(3):
            try:
                payload = {"model": model_name, "messages": messages, "temperature": 0.7, "max_tokens": 4096}
                resp = requests.post(self.api_url, headers=headers, json=payload, timeout=240)
                if resp.status_code!=200: raise Exception(str(resp.status_code))
                res = resp.json()['choices'][0]['message']['content']
                if validator:
                    if not validator(res)[0]: continue
                return res
            except: time.sleep(3)
        return "[FAILED]"

# =============================================================================
# 🚀 启动器 (支持12种语言选择，6x2对称布局)
# =============================================================================

class LanguageLauncher:
    def __init__(self):
        self.root = tk.Tk(); self.root.title("Language Selection"); 
        w,h=550,550; x,y=(self.root.winfo_screenwidth()-w)//2, (self.root.winfo_screenheight()-h)//2
        self.root.geometry(f"{w}x{h}+{x}+{y}"); self.selected_lang=None
        ttk.Label(self.root, text="Select Interface Language\n请选择界面语言", font=("Arial",14), justify=tk.CENTER).pack(pady=20)
        f=ttk.Frame(self.root); f.pack(pady=10)
        
        # 12 种语言，形成 6x2 矩阵
        langs=[
            ("中文 (Chinese)",'zh'),("English",'en'),
            ("日本語 (Japanese)",'ja'),("한국어 (Korean)",'ko'),
            ("བོད་ཡིག (Tibetan)",'bo'),("Pāḷi (Roman)",'pali'),
            ("Español (Spanish)",'es'),("Русский (Russian)",'ru'),
            ("Français (French)",'fr'),("Deutsch (German)",'de'),
            ("हिन्दी (Hindi)",'hi'),("Norsk (Norwegian)",'no')
        ]
        
        for i,(n,c) in enumerate(langs): 
            ttk.Button(f,text=n,command=lambda x=c:self.sel(x),width=22).grid(row=i//2,column=i%2,padx=10,pady=10)
    def sel(self,c): self.selected_lang=c; self.root.destroy()
    def run(self): self.root.mainloop(); return self.selected_lang

# =============================================================================
# 🖥️ GUI 主程序
# =============================================================================

class ZenUniversalApp:
    def __init__(self, root, ui_lang='zh'):
        self.root = root; self.ui_lang = ui_lang
        self.T = UI_STRINGS.get(ui_lang, UI_STRINGS['en'])
        self.root.title(self.T['app_title'])
        self.root.geometry("1200x950") # 稍微增加高度以容纳更多语言
        self.processor = ZenTextProcessor()
        self.vars_trans = {}; self.vars_exe = {}
        self.is_running = False; self.stop_event = threading.Event()
        self._setup_ui()
        
    def _setup_ui(self):
        top_container = ttk.Frame(self.root, padding=10); top_container.pack(fill=tk.X)
        api_group = ttk.LabelFrame(top_container, text=self.T['group_api'], padding=10)
        api_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        f1 = ttk.Frame(api_group); f1.pack(fill=tk.X, pady=2)
        ttk.Label(f1, text=self.T['lbl_url'], width=10).pack(side=tk.LEFT)
        self.api_url = tk.StringVar(value=DEFAULT_API_URL); ttk.Entry(f1, textvariable=self.api_url).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        ttk.Label(f1, text=self.T['lbl_key'], width=8).pack(side=tk.LEFT)
        self.api_key = tk.StringVar(value=DEFAULT_API_KEY); ttk.Entry(f1, textvariable=self.api_key, show="*", width=20).pack(side=tk.LEFT, padx=5)
        f2 = ttk.Frame(api_group); f2.pack(fill=tk.X, pady=5)
        ttk.Label(f2, text=self.T['lbl_model'], width=10).pack(side=tk.LEFT)
        self.model_name = tk.StringVar(value=DEFAULT_MODEL_NAME); ttk.Entry(f2, textvariable=self.model_name).pack(side=tk.LEFT, fill=tk.X, expand=True, padx=5)
        
        action_frame = ttk.Frame(top_container, padding=(10, 0, 0, 0)); action_frame.pack(side=tk.RIGHT, fill=tk.Y)
        style = ttk.Style(); style.configure("Big.TButton", font=("Arial", 12, "bold")); style.configure("Stop.TButton", font=("Arial", 10))
        self.btn_start = ttk.Button(action_frame, text=self.T['btn_start'], command=self.start, style="Big.TButton", width=15)
        self.btn_start.pack(side=tk.TOP, fill=tk.BOTH, expand=True, pady=(5, 5), ipady=10)
        self.btn_stop = ttk.Button(action_frame, text=self.T['btn_stop'], command=self.stop, style="Stop.TButton")
        self.btn_stop.pack(side=tk.BOTTOM, fill=tk.X, pady=(0, 5))

        mid_frame = ttk.Frame(self.root, padding=10); mid_frame.pack(fill=tk.X)
        file_group = ttk.LabelFrame(mid_frame, text=self.T['group_file'], padding=10)
        file_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        f_box = ttk.Frame(file_group); f_box.pack(fill=tk.X)
        self.file_path = tk.StringVar(); ttk.Entry(f_box, textvariable=self.file_path).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(f_box, text=self.T['btn_select_file'], command=self._select_file).pack(side=tk.LEFT, padx=5)
        t_box = ttk.Frame(file_group); t_box.pack(fill=tk.X, pady=5)
        ttk.Label(t_box, text=self.T['lbl_threshold']).pack(side=tk.LEFT)
        self.ts_limit = tk.IntVar(value=DEFAULT_MAX_TOKENS); ttk.Entry(t_box, textvariable=self.ts_limit, width=8).pack(side=tk.LEFT, padx=5)
        
        lang_group = ttk.LabelFrame(mid_frame, text=self.T['group_lang'], padding=10)
        lang_group.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        ttk.Label(lang_group, text=self.T['col_lang'], font=("Arial", 9, "bold")).grid(row=0, column=0, sticky=tk.W)
        ttk.Label(lang_group, text=self.T['col_trans'], font=("Arial", 9, "bold")).grid(row=0, column=1)
        ttk.Label(lang_group, text=self.T['col_exe'], font=("Arial", 9, "bold")).grid(row=0, column=2)
        
        # 12 种语言列表
        order = ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']
        for i, key in enumerate(order):
            row = i + 1
            ttk.Label(lang_group, text=self.T.get(f'lang_{key}', key)).grid(row=row, column=0, sticky=tk.W, padx=5, pady=2)
            val_trans = False; val_exe = (key == self.ui_lang) 
            vt = tk.BooleanVar(value=val_trans); self.vars_trans[key] = vt
            ve = tk.BooleanVar(value=val_exe); self.vars_exe[key] = ve
            ttk.Checkbutton(lang_group, variable=vt).grid(row=row, column=1)
            ttk.Checkbutton(lang_group, variable=ve).grid(row=row, column=2)

        main_content = ttk.PanedWindow(self.root, orient=tk.VERTICAL); main_content.pack(fill=tk.BOTH, expand=True, padx=10, pady=(0, 10))
        log_frame = ttk.LabelFrame(main_content, text=self.T['group_log']); main_content.add(log_frame, weight=1)
        self.log_text = scrolledtext.ScrolledText(log_frame, height=8, font=("Consolas", 9), state='normal'); self.log_text.pack(fill=tk.BOTH, expand=True)
        preview_frame = ttk.LabelFrame(main_content, text=self.T['group_preview']); main_content.add(preview_frame, weight=4)
        self.preview_area = scrolledtext.ScrolledText(preview_frame, font=("微软雅黑", 11)); self.preview_area.pack(fill=tk.BOTH, expand=True)
        self.progress = ttk.Progressbar(self.root, mode='determinate'); self.progress.pack(fill=tk.X, padx=10, pady=(0, 5))

    def _select_file(self):
        f = filedialog.askopenfilename(filetypes=[("Text Files", "*.txt")])
        if f: self.file_path.set(f)
    def log(self, msg):
        self.log_text.insert(tk.END, f"[{datetime.now().strftime('%H:%M:%S')}] {msg}\n"); self.log_text.see(tk.END)
    def stop(self):
        if self.is_running: self.stop_event.set(); self.log(self.T['msg_stop'])

    def start(self):
        if not self.file_path.get(): messagebox.showerror("Error", self.T['msg_err_nofile']); return
        selected_tasks = []
        selected_langs = set()
        # 遍历 12 种语言
        for lang in ['zh', 'en', 'ja', 'ko', 'bo', 'pali', 'es', 'ru', 'fr', 'de', 'hi', 'no']:
            if self.vars_trans[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'trans'})
                selected_langs.add(lang)
            if self.vars_exe[lang].get(): 
                selected_tasks.append({'lang': lang, 'type': 'exe'})
                selected_langs.add(lang)
        if not selected_tasks: messagebox.showerror("Error", self.T['msg_err_notask']); return
        self.is_running = True; self.stop_event.clear(); self.btn_start.config(state=tk.DISABLED)
        threading.Thread(target=self._run_process, args=(selected_tasks, list(selected_langs)), daemon=True).start()

    def _run_process(self, tasks, active_langs):
        input_file = self.file_path.get()
        base_name = os.path.splitext(input_file)[0]
        try:
            self.log(self.T['msg_read_start'])
            with open(input_file, 'r', encoding='utf-8') as f: content = f.read()
            segments = self.processor.smart_segmentation(content, self.ts_limit.get())
            total_segs = len(segments)
            self.log(self.T['msg_seg_done'].format(total_segs))
            
            split_file_path = f"{base_name}_split.txt"
            with open(split_file_path, 'w', encoding='utf-8') as f:
                for seg in segments: f.write(f"【{seg['title']}】\n{seg['content']}\n\n{'='*40}\n\n")
            self.log(self.T['msg_split_created'].format(os.path.basename(split_file_path)))

            handles = {}
            for t in tasks:
                suffix = LangConfig.get_file_suffix(t['lang'], t['type'])
                out_path = base_name + suffix
                f = open(out_path, 'w', encoding='utf-8')
                f.write(f"=== {t['type'].upper()} ({t['lang'].upper()}) ===\nSource: {os.path.basename(input_file)}\n\n")
                handles[(t['lang'], t['type'])] = f
                self.log(self.T['msg_file_created'].format(os.path.basename(out_path)))
            
            for lang in active_langs:
                suffix = LangConfig.get_file_suffix(lang, 'combined')
                out_path = base_name + suffix
                f = open(out_path, 'w', encoding='utf-8')
                f.write(f"=== CONTRAST VERSION ({lang.upper()}) ===\nSource: {os.path.basename(input_file)}\n\n")
                handles[(lang, 'combined')] = f
                self.log(self.T['msg_file_created'].format(os.path.basename(out_path)))

            ai = AiEngine(self.api_url.get(), self.api_key.get())
            for i, seg in enumerate(segments):
                if self.stop_event.is_set(): break
                title, text = seg['title'], seg['content']
                self.log(self.T['msg_processing'].format(i+1, total_segs, title))
                segment_results = {lang: {'trans': None, 'exe': None} for lang in active_langs}

                for t in tasks:
                    if self.stop_event.is_set(): break
                    lang, type_ = t['lang'], t['type']
                    prompt = LangConfig.get_trans_prompt(lang) if type_ == 'trans' else LangConfig.get_exe_prompt(lang)
                    validator = LangConfig.validate_pali if lang == 'pali' else None
                    
                    lang_name = self.T.get(f'lang_{lang}', lang)
                    task_display = self.T['col_trans'] if type_ == 'trans' else self.T['col_exe']
                    self.log(self.T['msg_generating'].format(lang_name, task_display))
                    
                    result = ai.process(title, text, prompt, self.model_name.get(), validator)
                    handles[(lang, type_)].write(f"【{title}】\n{result}\n\n{'='*60}\n\n"); handles[(lang, type_)].flush()
                    segment_results[lang][type_] = result
                    
                    self.preview_area.delete(1.0, tk.END)
                    self.preview_area.insert(tk.END, f"--- {lang_name} [{type_}] ---\n{title}\n\n{result}")
                
                for lang in active_langs:
                    combined_f = handles[(lang, 'combined')]
                    combined_text = f"【{title}】\n\n--- Source ---\n{text}\n"
                    res_trans = segment_results[lang].get('trans')
                    res_exe = segment_results[lang].get('exe')
                    if res_trans: combined_text += f"\n--- Translation ---\n{res_trans}\n"
                    if res_exe: combined_text += f"\n--- Exegesis ---\n{res_exe}\n"
                    combined_text += f"\n{'='*60}\n\n"
                    combined_f.write(combined_text); combined_f.flush()

                self.progress['value'] = ((i + 1) / total_segs) * 100
            
            self.log("DONE!")
            messagebox.showinfo(self.T['msg_done_title'], self.T['msg_done_body'])
        except Exception as e:
            self.log(self.T['err_fatal'].format(str(e))); messagebox.showerror("Error", str(e))
        finally:
            if 'handles' in locals():
                for f in handles.values(): f.close()
            self.is_running = False; self.btn_start.config(state=tk.NORMAL); self.progress['value'] = 0

if __name__ == "__main__":
    launcher = LanguageLauncher()
    selected_lang = launcher.run()
    if selected_lang:
        root = tk.Tk()
        app = ZenUniversalApp(root, ui_lang=selected_lang)
        root.mainloop()
